package gov.va.genisis2.dao.impl;

import gov.va.genisis2.dao.ICommentHistoryDao;
import gov.va.genisis2.model.CommentHistory;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.query.Query;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.hibernate5.HibernateTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class CommentHistoryDao.
 * 
 * The CommentHistory data access object (DAO) is an object that provides an abstract
 * interface to some type of database or other persistence mechanism. 
 * By mapping application calls to the persistence layer, CommentHistory
 * DAO provide some specific data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class CommentHistoryDao implements ICommentHistoryDao {

	/** The logger. */
	private final org.slf4j.Logger logger = LoggerFactory.getLogger(CommentHistoryDao.class);

	/** The hibernate template. */
	private HibernateTemplate hibernateTemplate;

	/**
	 * This method is used to get hibernate template.
	 * @return HibernateTemplate This returns hibernateTemplate.
	 */
	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}

	/**
	 * This method is used to set hibernate template.
	 * @param hibernateTemplate  The hibernateTemplate.
	 */
	@Autowired
	public void setHibernateTemplate(HibernateTemplate hibernateTemplate) {
		this.hibernateTemplate = hibernateTemplate;
	}

	/**
	 * This method is used to create commentHistory.
	 * @param entity The entity.
	 * @return int This returns created comment id.
	 */
	@Override
	public int createCommentHistory(CommentHistory entity) {
		int id = 0;
		if (entity == null) {
			return 0;
		}
		entity.setCommentId(this.maxRowValue());

		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		EntityTransaction transaction = null;
		try {
			transaction = entityManager.getTransaction();
			transaction.begin();
			entityManager.persist(entity);
			transaction.commit();
			id = entity.getCommentId();
		} catch (RuntimeException e) {
			logger.error("RequestDao:  Comment History Creation failed. Transaction Rolled back.", e);
			if (transaction != null)
				transaction.rollback();
		} finally {
			entityManager.close();
		}
		return id;
	}

	/**
	 * This method is used to updated commentHistory.
	 * @param entity The entity.
	 * @return int This returns updated comment id.
	 */
	@Override
	public int updateCommentHistory(CommentHistory entity) {
		hibernateTemplate.saveOrUpdate(entity);
		return entity.getCommentId();
	}

	/**
	 *This method is used to get commentHistory by id.
	 *@param id The Id.
	 *@return CommentHistory This returns commentHistory.
	 */
	@Override
	public CommentHistory getCommentHistoriesByID(int id) {
		return hibernateTemplate.get(CommentHistory.class, id);
	}

	/**
	 * This method is used to get list of commentHistory by UID.
	 * @param uid The uid.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoriesByUID(String uid) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
		Root<CommentHistory> root = query.from(CommentHistory.class);
		query.where(criteriaBuilder.equal(root.get("createdBy"), uid));
		query.select(root);
		return entityManager.createQuery(query).getResultList();
	}

	/** 
	 * This method is used to getCommentHistories.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistories() {
		return (List<CommentHistory>) hibernateTemplate.find("from CommentHistory");
	}

	/**
	 * This method is used to get max value record from the table.
	 * @return int This returns max value record plus one.
	 */
	@SuppressWarnings("unchecked")
	private int maxRowValue() {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(CommentHistory.class);
		Root root = criteriaQuery.from(CommentHistory.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("id")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();
		return result.intValue() + 1;
	}

	/** 
	 * This method is used to submit or modify the table.
	 * @param entity The entity.
	 * @return int This returns modified commentHistory id.
	 */
	@Override
	public int submitOrModify(CommentHistory entity) {
		int maxRowCommentHistory = maxRowValueCommentHistory();
		entity.setCommentId(maxRowCommentHistory);
		hibernateTemplate.saveOrUpdate(entity);
		return entity.getCommentId();
	}

	/**
	 * This method is used to get list of commentHistories by id.
	 * @param requestId The requestId.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoryById(int requestId)  {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<CommentHistory> criteria = builder.createQuery(CommentHistory.class);
		Root<CommentHistory> org = criteria.from(CommentHistory.class);
		criteria.select(org);
		criteria.where(builder.and(builder.equal(org.get("request").get("id"), requestId)));
		criteria.orderBy(builder.desc(org.get("createdOn")));
		return entityManager.createQuery(criteria).getResultList();
	}

	/**
	 * This method is used to get commentHistory by comment id.
	 * @param commentId The commentId.
	 * @return CommentHistory This returns commentHistory.
	 */
	@Override
	public CommentHistory getCommentHistoryByCommentId(int commentId)  {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<CommentHistory> criteria = builder.createQuery(CommentHistory.class);
		Root<CommentHistory> org = criteria.from(CommentHistory.class);
		criteria.select(org);
		criteria.where(builder.and(builder.equal(org.get("commentId"), commentId)));
		criteria.orderBy(builder.desc(org.get("createdOn")));
		return  entityManager.createQuery(criteria).getSingleResult();
	}

	/**
	 * This method is used to get list of commentHistories by request id and status id.
	 * @param requestId   The requestId.
	 * @param taskId  The taskId.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoryByRequestIdStatusId(int requestId, String taskId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<CommentHistory> criteria = builder.createQuery(CommentHistory.class);
		Root<CommentHistory> org = criteria.from(CommentHistory.class);

		Predicate condition1 = builder.equal(org.get("request").get("id"), requestId);

		Predicate condition2 = builder.equal(org.get("taskId"), taskId);
		Predicate conditions = builder.and(condition1, condition2);
		criteria.where(conditions);
		List<CommentHistory> result = null;
		Query query = (Query) entityManager.createQuery(criteria);
		try {
			result = query.getResultList();
		} catch (NoResultException e) {
			logger.error("Request Id : " + requestId + "  Error occured while generating CommentHistory ", e);
		}
		return result;
	}

	/**
	 * This method is used to get max value record from the table.
	 * @return int This returns max value record plus one.
	 */

	private int maxRowValueCommentHistory() {
		int maxRowVal = 1;
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(CommentHistory.class);
		Root root = criteriaQuery.from(CommentHistory.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("commentId")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();

		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}

}